// This version of navengine.js was modified to allow prevention
// of the persistence of global objectives states, node scores,
// and all node state properties except completion.  The change
// was made to facilitate reducing the amount of suspend_data being used.
// The trackAllData variable is used to turn the new "limited
// tracking" feature on and off.

var trackAllData = true;

var DEFAULT_LOAD_STRING_SEPARATOR = "$";
var OBJ_HASH_LENGTH = 5;
var NODE_HASH_LENGTH = 51;
var NAV_DELAY = 1;

function NavEngine()
{
	// Array to hold references to 
	// each tree in the forest.
	this.forest = null;
	
	this.title = null;
	this.description = null;
	
	// HashTable to hold refernces to
	// all nodes in the lesson, indexed
	// by the node's id.
	this.nodes = null;
	this.taggedNodes = null;
	this.numberedNodes = null;
	this.seqEngine = null;
	this.objectives = null;
	this.objHash = null;
	this.currentNode = null;
	
	// Holds a reference to a blocked node, if one
	// is encountered during a canNav(), getNav()
	// or doNav() call.
	this.blockedNode = null;
	
	this.navWin = null;
	this.illegalNavMessage = null;
	this.events = new EventsObject("navstatechange","navwindow","reorder","choice","flow","forwardOnly","navchange","nodecompleted","nodeattempted","nodescore","nodesatisfied","nodeblock","nodeskip","nodeenabled","nodevisible","nodechoice","nodeflow","nodeforwardonly");
	this.cFlow = null;
	this.cChoice = null;
	this.cForwardOnly = null;
	this.loadStringSeparator = null;
	this.mapAPI = null;
	this.nodeAPI = null;
	
	this.setDefaultValues();

	this.registerEvent("navstatechange","navStateChangeCallBack", this,null,true);
	
	this.htAncestorsEnabled = null;
	this.htNodeToRoot = null;
	this.arrBlockedNodes = null;
	
	this.quickReferenceContainersDirty = true;
}
// Private & Friend API
NavEngine.prototype.populateQuickReferenceContainers = NavEnginePopulateQuickReferenceContainers;
NavEngine.prototype.navStateChangeCallBack = NavEngineNavStateChangeCallBack;
NavEngine.prototype.setDefaultValues = NavEngineSetDefaultValues;
NavEngine.prototype.getObjectiveById = NavEngineGetObjectiveById;
NavEngine.prototype.initForest = NavEngineInitForest;
NavEngine.prototype.resetForest = NavEngineResetForest;
NavEngine.prototype.getTreeHash = NavEngineGetTreeHash;
NavEngine.prototype.getTreeHashRec = NavEngineGetTreeHashRec;
NavEngine.prototype.getBlockedNodes = NavEngineGetBlockedNodes;
NavEngine.prototype.getBlockedNodesRec = NavEngineGetBlockedNodesRec;
NavEngine.prototype.populateNodesTable = NavEnginePopulateNodesTable;
NavEngine.prototype.populateNodesTableRec = NavEnginePopulateNodesTableRec;
NavEngine.prototype.numberNodes = NavEngineNumberNodes;
NavEngine.prototype.numberNodesRec = NavEngineNumberNodesRec;
NavEngine.prototype.callbackReorder = NavEngineCallbackReorder;	
NavEngine.prototype.firstNodeInSequence = NavEngineFirstNodeInSequence;	
NavEngine.prototype.nextNodeInSequence = NavEngineNextNodeInSequence;	
NavEngine.prototype.previousNodeInSequence = NavEnginePreviousNodeInSequence;
NavEngine.prototype.clearParentStacks = NavEngineClearParentStacks;
NavEngine.prototype.clearParentStacksRec = NavEngineClearParentStacksRec;
NavEngine.prototype.isRootNode = NavEngineIsRootNode;
NavEngine.prototype.callbackNodeAttempted = NavEngineCallbackNodeAttempted;
NavEngine.prototype.callbackNodeCompleted = NavEngineCallbackNodeCompleted;
NavEngine.prototype.callbackNodeScore = NavEngineCallbackNodeScore;
NavEngine.prototype.callbackNodeSatisfied = NavEngineCallbackNodeSatisfied;
NavEngine.prototype.callbackNodeBlock = NavEngineCallbackNodeBlock;
NavEngine.prototype.callbackNodeSkip = NavEngineCallbackNodeSkip;
NavEngine.prototype.callbackNodeChoice = NavEngineCallbackNodeChoice;
NavEngine.prototype.callbackNodeFlow = NavEngineCallbackNodeFlow;
NavEngine.prototype.callbackNodeForwardOnly = NavEngineCallbackNodeForwardOnly;
NavEngine.prototype.callbackNodeEnabled = NavEngineCallbackNodeEnabled;
NavEngine.prototype.callbackNodeVisible = NavEngineCallbackNodeVisible;
NavEngine.prototype.registerNodeEvents = NavEngineRegisterNodeEvents;
NavEngine.prototype.registerNodeEventsRec = NavEngineRegisterNodeEventsRec;
NavEngine.prototype.setCurrentNode = NavEngineSetCurrentNode;
NavEngine.prototype.cloneForest = NavEngineCloneForest;
NavEngine.prototype.cloneNode = NavEngineCloneNode;
NavEngine.prototype.resolveClonedNodeReferencesRec = NavEngineResolveClonedNodeReferencesRec;
NavEngine.prototype.populateNodesPropertiesRec = NavEnginePopulateNodesPropertiesRec;
NavEngine.prototype.restoreNodesProperties = NavEngineRestoreNodesProperties;
NavEngine.prototype.restoreNodesPropertiesRec = NavEngineRestoreNodesPropertiesRec;
NavEngine.prototype.getNodeByNum = NavEngineGetNodeByNum;
NavEngine.prototype.resolveNodeRef = NavEngineResolveNodeRef;
NavEngine.prototype.getLastNodeInSubtree = NavEngineGetLastNodeInSubtree;

// Public API
NavEngine.prototype.initNav = NavEngineInitNav;
NavEngine.prototype.canNav = NavEngineCanNav;
NavEngine.prototype.evalNav = NavEngineEvalNav;
NavEngine.prototype.getNav = NavEngineGetNav;
NavEngine.prototype.doNav = NavEngineDoNav;
NavEngine.prototype.toString = NavEngineToString;
NavEngine.prototype.getNodeById = NavEngineGetNodeById;
NavEngine.prototype.getCurrentNode = NavEngineGetCurrentNode;
NavEngine.prototype.showCurrent = NavEngineShowCurrent;
NavEngine.prototype.showNodeInWindow = NavEngineShowNodeInWindow;
NavEngine.prototype.canGotoNode = NavEngineCanGotoNode;
NavEngine.prototype.evalGotoNode = NavEngineEvalGotoNode;
NavEngine.prototype.gotoNode = NavEngineGotoNode;
NavEngine.prototype.gosubNode = NavEngineGosubNode;
NavEngine.prototype.registerEvent = RegisterEvent;
NavEngine.prototype.unregisterEvent = UnregisterEvent;
NavEngine.prototype.loadFromXMLDoc = NavEngineLoadFromXMLDoc;
NavEngine.prototype.getChoice = NavEngineGetChoice;
NavEngine.prototype.getFlow = NavEngineGetFlow;
NavEngine.prototype.getForwardOnly = NavEngineGetForwardOnly;
NavEngine.prototype.resetNode = NavEngineResetNode;
NavEngine.prototype.setNodeScore = NavEngineSetNodeScore;
NavEngine.prototype.getNodeScore = NavEngineGetNodeScore;
NavEngine.prototype.setNodeAttempted = NavEngineSetNodeAttempted;
NavEngine.prototype.getNodeAttempted = NavEngineGetNodeAttempted;
NavEngine.prototype.setNodeCompleted = NavEngineSetNodeCompleted;
NavEngine.prototype.getNodeCompleted = NavEngineGetNodeCompleted;
NavEngine.prototype.setNodeSatisfied = NavEngineSetNodeSatisfied;
NavEngine.prototype.getNodeSatisfied = NavEngineGetNodeSatisfied;
NavEngine.prototype.setNodeSkip = NavEngineSetNodeSkip;
NavEngine.prototype.getNodeSkip = NavEngineGetNodeSkip;
NavEngine.prototype.setNodeChoice = NavEngineSetNodeChoice;
NavEngine.prototype.getNodeChoice = NavEngineGetNodeChoice;
NavEngine.prototype.setNodeFlow = NavEngineSetNodeFlow;
NavEngine.prototype.getNodeFlow = NavEngineGetNodeFlow;
NavEngine.prototype.setNodeForwardOnly = NavEngineSetNodeForwardOnly;
NavEngine.prototype.getNodeForwardOnly = NavEngineGetNodeForwardOnly;
NavEngine.prototype.setNodeBlock = NavEngineSetNodeBlock;
NavEngine.prototype.getNodeBlock = NavEngineGetNodeBlock;
NavEngine.prototype.setNodeEnabled = NavEngineSetNodeEnabled;
NavEngine.prototype.getNodeEnabled = NavEngineGetNodeEnabled;
NavEngine.prototype.setNodeVisible = NavEngineSetNodeVisible;
NavEngine.prototype.getNodeVisible = NavEngineGetNodeVisible;
NavEngine.prototype.getNavWindow = NavEngineGetNavWindow;
NavEngine.prototype.setNavWindow = NavEngineSetNavWindow;
NavEngine.prototype.getNodeId = NavEngineGetNodeId;
NavEngine.prototype.getNodeUrl = NavEngineGetNodeUrl;
NavEngine.prototype.getNodeTitle = NavEngineGetNodeTitle;
NavEngine.prototype.getNodeTag = NavEngineGetNodeTag;
NavEngine.prototype.clone = NavEngineClone;
NavEngine.prototype.getIllegalNavMessage = NavEngineGetIllegalNavMessage;
NavEngine.prototype.setIllegalNavMessage = NavEngineSetIllegalNavMessage;
NavEngine.prototype.clone = NavEngineClone;
NavEngine.prototype.load = NavEngineLoad;
NavEngine.prototype.dump = NavEngineDump;
NavEngine.prototype.setLoadStringSeparator = NavEngineSetLoadStringSeparator;
NavEngine.prototype.getLoadStringSeparator = NavEngineGetLoadStringSeparator;
NavEngine.prototype.getLevel = NavEngineGetLevel;
NavEngine.prototype.getNodeChildIds = NavEngineGetNodeChildIds;

function NavEngineResolveNodeRef(id)
{
	var nodeRef = null;
	
	if (typeof(id) == "string")
	{	
		nodeRef = this.navEngine.getNodeById(id);
	}
	else
	{
		if (typeof(id) == "object" && id.constructor == NavNode)
		{
			nodeRef = id;
		}
	}
	
	return nodeRef;
}

function NavEngineGetLevel(id, contentOnlyFlag)
{
	if (contentOnlyFlag == null)
	{
		contentOnlyFlag = false;
	}

	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		
	}
	else
	{
		return -1;
	}
}

function NavEngineSetLoadStringSeparator(separator)
{
	this.loadStringSeparator = separator;
}

function NavEngineGetLoadStringSeparator()
{
	return this.loadStringSeparator;
}

function NavEngineLoad(strVal)
{
	// Strip-off the leading character, which
	// is used as the separator.
	var separator = strVal.charAt(0);
	strVal = strVal.slice(1,strVal.length);
	
	// Split at separator into an array
	var values = strVal.split(separator);
	
	// Split values up into their individual components
	var controls = new BooleanArray(values[0]);
	var currentNodeId = values[1];
	var blockedNodeId = values[2];
	var objectivesSatisfieds = new BooleanArray(values[3]);
	var objectivesScores = null;
	if (values[4] == "")
	{
		objectivesScores = new Array();
	}
	else
	{
		objectivesScores = values[4].split(",");
	}
	var nodePropBolArr = new BooleanArray(values[5]);
	var nodeScores = null;
	if (values[6] == "")
	{
		nodeScores = new Array();
	}
	else
	{
		nodeScores = values[6].split(",");
	}
	var nodeParentStacks = null;
	if (values[7] == "")
	{
		nodeParentStacks = new Array();
	}
	else
	{
	 	nodeParentStacks = values[7].split(";");
	}
	var nodeChildArrays = null;
	if (values[8] == "")
	{
		nodeChildArrays = new Array();
	}
	else
	{
		nodeChildArrays = values[8].split(";");
	}
	
	// Make assignments to the NavEngine and NavNode Objects.
	this.cFlow = controls.get(0);
	this.cChoice = controls.get(1);
	this.cForwardOnly = controls.get(2);
	this.currentNode = this.getNodeById(currentNodeId);
	this.blockedNode = this.getNodeById(blockedNodeId);
	for (var i=0;i<this.objectives.length;i++)
	{
		this.objectives[i].satisfied = objectivesSatisfieds.get(i);
		this.objectives[i].score = floatParse(objectivesScores[i],SCORE_PRECISION);
	}
	this.restoreNodesProperties(nodePropBolArr, this.forest[i]);
	for (var i=0;i<nodeScores.length;i++)
	{
		var parts = nodeScores[i].split(":");
		this.getNodeByNum(parts[0]).objective.score = floatParse(parts[1],SCORE_PRECISION);
	}
	for (var i=0;i<nodeParentStacks.length;i++)
	{
		var parts = nodeParentStacks[i].split(":");
		parts[1] = parts[1].split(",");
		for (var t=0;t<parts[1].length;t++)
		{
			this.getNodeById(parts[0]).parentStack[this.getNodeById(parts[0]).parentStack.length] = this.getNodeById(parts[1][t]);
		}
	}
	for (var i=0;i<nodeChildArrays.length;i++)
	{
		var parts = nodeChildArrays[i].split(":");
		parts[1] = parts[1].split(",");
		for (var t=0;t<parts[1].length;t++)
		{
			this.getNodeById(parts[0]).selectedChildren[this.getNodeById(parts[0]).selectedChildren.length] = this.getNodeById(parts[1][t]);
		}
	}
	this.numberNodes();

	// These lines of code were added since it's possible that when
	// loading lesson data from a previous session that the
	// current page (this.currentNode) may not be accessible.
	if (this.currentNode && !this.canGotoNode(this.currentNode.id))
	{
		this.setCurrentNode(null);
	}
}

var restoreCount = 0;
function NavEngineRestoreNodesProperties(boolArr)
{
	restoreCount = 0;
	for (var i=0;i<this.forest.length;i++)
	{
		this.forest[i].parentStack.length = 0;
		this.restoreNodesPropertiesRec(boolArr, this.forest[i]);
	}
}

function NavEngineRestoreNodesPropertiesRec(boolArr, nodeRef)
{
	if (trackAllData)
	{
		nodeRef.skip = boolArr.get(restoreCount++);
		nodeRef.cChoice = boolArr.get(restoreCount++);
		nodeRef.cFlow = boolArr.get(restoreCount++);
		nodeRef.cForwardOnly = boolArr.get(restoreCount++);
		nodeRef.block = boolArr.get(restoreCount++);
		nodeRef.enabled = boolArr.get(restoreCount++);
		nodeRef.visible = boolArr.get(restoreCount++);
	}
	nodeRef.completed = boolArr.get(restoreCount++);
	if (trackAllData)
	{
		nodeRef.attempted = boolArr.get(restoreCount++);
		nodeRef.objective.satisfied = boolArr.get(restoreCount++);
	}
	
	if (nodeRef.parentStack.length > 1)
	{
		nodeRef.parentStack.length = 1;
	}
	nodeRef.selectedChildren.length = 0;
	
	for (var i=0;i<nodeRef.children.length;i++)
	{
		this.restoreNodesPropertiesRec(boolArr, nodeRef.children[i])
	}
}

function NavEngineDump()
{
	var blankArray = new Array();
	var blankBoolArray = new BooleanArray();

	var s = this.loadStringSeparator;
	
	var controls = new BooleanArray();
	controls.set(0,this.cFlow);
	controls.set(1,this.cChoice);
	controls.set(2,this.cForwardOnly);
	
	var currentNodeId = ((this.currentNode==null)?(""):(this.currentNode.id))
	var blockedNodeId = ((this.blockedNode==null)?(""):(this.blockedNode.id))
	
	var objectivesSatisfieds = new BooleanArray();
	var objectivesScores = new Array();
	for (var i=0;i<this.objectives.length;i++)
	{
		objectivesSatisfieds.set(i,this.objectives[i].satisfied);
		objectivesScores[i] = (this.objectives[i].score + "");
	}
		
	var nodePropBoolArr = new BooleanArray();
	var nodeScores = new Array();
	var nodeParentStacks = new Array();
	var nodeChildArrays = new Array();
	for (var i=0;i<this.forest.length;i++)
	{
		this.populateNodesPropertiesRec(nodePropBoolArr, nodeScores, nodeParentStacks, nodeChildArrays, this.forest[i]);
	}
	
	return s + controls.dump() + s + currentNodeId + s + blockedNodeId + s + ((trackAllData)?(objectivesSatisfieds.dump()):(blankBoolArray.dump())) + s + ((trackAllData)?(objectivesScores.join(",")):(blankArray.join(","))) + s + nodePropBoolArr.dump() + s + ((trackAllData)?(nodeScores.join(",")):(blankArray.join(","))) + s + nodeParentStacks.join(";") + s + nodeChildArrays.join(";");
}

function NavEnginePopulateNodesPropertiesRec(boolArr, scoreArray, parentArray, childArray, nodeRef)
{
	if (trackAllData)
	{
		boolArr.set(boolArr.getLength(), nodeRef.skip);
		boolArr.set(boolArr.getLength(), nodeRef.cChoice);
		boolArr.set(boolArr.getLength(), nodeRef.cFlow);
		boolArr.set(boolArr.getLength(), nodeRef.cForwardOnly);
		boolArr.set(boolArr.getLength(), nodeRef.block);
		boolArr.set(boolArr.getLength(), nodeRef.enabled);
		boolArr.set(boolArr.getLength(), nodeRef.visible);
	}
	boolArr.set(boolArr.getLength(), nodeRef.completed);
	if (trackAllData)
	{
		boolArr.set(boolArr.getLength(), nodeRef.attempted);
		boolArr.set(boolArr.getLength(), nodeRef.objective.satisfied);
	}

	if (nodeRef.objective.score != 0)
	{
		scoreArray[scoreArray.length] = nodeRef.num + ":" + nodeRef.objective.score;
	}
	
	if (nodeRef.parentStack.length > 1)
	{
		var tempArr = nodeRef.parentStack.slice(1,nodeRef.parentStack.length);
		for (var i=0;i<tempArr.length;i++)
		{
			tempArr[i] = tempArr[i].id;
		}
		parentArray[parentArray.length] = nodeRef.id + ":" + tempArr.join(",");
	}
	
	if (nodeRef.selectedChildren.length > 0)
	{
		var tempArr = nodeRef.selectedChildren.slice(0,nodeRef.selectedChildren.length);
		for (var i=0;i<tempArr.length;i++)
		{
			tempArr[i] = tempArr[i].id;
		}
		childArray[childArray.length] = nodeRef.id + ":" + tempArr.join(",");
	}
	
	for (var i=0;i<nodeRef.children.length;i++)
	{
		this.populateNodesPropertiesRec(boolArr, scoreArray, parentArray, childArray, nodeRef.children[i])
	}
}

function NavEngineClone(bolShareForest)
{
	var retVal;

	if (bolShareForest == null)
	{
		bolShareForest = true;
	}

	if (bolShareForest == true)
	{
		retVal = new NavEngine();
		retVal.forest = this.forest;
		retVal.nodes = this.nodes;
		retVal.taggedNodes = this.taggedNodes;
		retVal.numberedNodes = this.numberedNodes;
		retVal.seqEngine = this.seqEngine;
		retVal.nodeAPI = this.nodeAPI;
		retVal.mapAPI = this.mapAPI;
		retVal.objHash = this.objHash;
		retVal.objectives = this.objectives;
		retVal.currentNode = this.currentNode;
		retVal.blockedNode = this.blockedNode;
		retVal.navWin = this.navWin;
		retVal.illegalNavMessage = this.illegalNavMessage;
		retVal.events = new EventsObject("navstatechange","navwindow","reorder","choice","flow","forwardOnly","navchange","nodecompleted","nodeattempted","nodescore","nodesatisfied","nodeblock","nodeskip","nodeenabled","nodevisible");
		retVal.cFlow = this.cFlow;
		retVal.cChoice = this.cChoice;
		retVal.cForwardOnly = this.cForwardOnly;
		retVal.title = this.title;
		retVal.description = this.description;
	}
	else
	{
		retVal = new NavEngine();
		retVal.forest = this.cloneForest();
		
		retVal.nodes = new HashTable(NODE_HASH_LENGTH);
		retVal.taggedNodes = new HashTable(NODE_HASH_LENGTH);
		retVal.numberedNodes = new HashTable(NODE_HASH_LENGTH);
		
		retVal.seqEngine = new NavSeqEngine(retVal);
		retVal.mapAPI = new NavMapAPI(retVal);
		retVal.nodeAPI = new NavNodeAPI(retVal);
		retVal.objHash = new HashTable(OBJ_HASH_LENGTH);
		retVal.objectives = new Array();
		var tempArr = this.objHash.getKeys();
		for (var i=0;i<tempArr.length;i++)
		{
			var tempObj1 = this.objHash.retrieve(tempArr[i].toLowerCase());
			var tempObj2 = new NavObjective();
			tempObj2.id = tempObj1.id;
			tempObj2.satisfied = tempObj1.satisfied;
			tempObj2.score = tempObj1.score;
			tempObj2.events = new EventsObject("score","satisfied");
			
			retVal.objHash.insert(tempObj2.id, tempObj2);
			retVal.objectives[retVal.objectives.length] = tempObj2;
		}
		
		retVal.navWin = this.navWin;
		retVal.illegalNavMessage = this.illegalNavMessage;
		retVal.events = new EventsObject("navstatechange","navwindow","reorder","choice","flow","forwardOnly","navchange","nodecompleted","nodeattempted","nodescore","nodesatisfied","nodeblock","nodeskip","nodeenabled","nodevisible");
		retVal.cFlow = this.cFlow;
		retVal.cChoice = this.cChoice;
		retVal.cForwardOnly = this.cForwardOnly;
		retVal.title = this.title;
		retVal.description = this.description;
		
		retVal.populateNodesTable();
		for (var i=0;i<retVal.forest.length;i++)
		{
			this.resolveClonedNodeReferencesRec(retVal, retVal.forest[i]);
			retVal.forest[i].resolveReferences(retVal);
		}
		retVal.numberNodes();
		
		retVal.registerNodeEvents();
		
		if (this.currentNode != null)
		{
			retVal.currentNode = retVal.getNodeById(this.currentNode.id);
		}
		if (this.blockedNode != null)
		{
			retVal.blockedNode = retVal.getNodeById(this.blockedNode.id);
		}
	}
	
	return retVal;
}

function NavEngineCloneForest()
{
	var retVal = new Array();
	for (var i=0;i<this.forest.length;i++)
	{
		retVal[i] = this.cloneNode(this.forest[i]);
	}
	
	return retVal;
}

function NavEngineCloneNode(nodeRef)
{
	var retVal = new NavNode();
	retVal.skip = nodeRef.skip;
	retVal.block = nodeRef.block;
	retVal.enabled = nodeRef.enabled;
	retVal.visible = nodeRef.visible;
	retVal.transparent = nodeRef.transparent;
	retVal.position = nodeRef.position.concat();
	
	retVal.skipDefault = nodeRef.skipDefault;
	retVal.blockDefault = nodeRef.blockDefault;
	retVal.enabledDefault = nodeRef.enabledDefault;
	retVal.visibleDefault = nodeRef.visibleDefault;
	retVal.choiceDefault = nodeRef.choiceDefault;
	retVal.flowDefault = nodeRef.flowDefault;
	retVal.forwardOnlyDefault = nodeRef.forwardOnlyDefault;
	
	retVal.completed = nodeRef.completed;
	retVal.attempted = nodeRef.attempted;
	
	retVal.objective = new NavObjective();
	retVal.objective.id = nodeRef.objective.id;
	retVal.objective.satisfied = nodeRef.objective.satisfied;
	retVal.objective.score = nodeRef.objective.score;
	retVal.objective.events = new EventsObject("satisfied", "score");
	
	retVal.parentStack = new Array();
	for (var i=0;i<nodeRef.parentStack.length;i++)
	{
		retVal.parentStack[i] = nodeRef.parentStack[i].id;
	}
	retVal.selectedChildren = new Array();
	for (var i=0;i<nodeRef.selectedChildren.length;i++)
	{
		retVal.selectedChildren[i] = nodeRef.selectedChildren[i].id;
	}
	
	retVal.id = nodeRef.id;
	retVal.num = nodeRef.num;
	retVal.url = nodeRef.url;
	retVal.tag = nodeRef.tag;
	retVal.title = nodeRef.title;
	retVal.weight = nodeRef.weight;

	retVal.children = new Array();
	for (var i=0;i<nodeRef.children.length;i++)
	{
		retVal.children[i] = this.cloneNode(nodeRef.children[i]);
	}
	
	retVal.objectiveMaps = new Array();
	for (i=0;i<nodeRef.objectiveMaps.length;i++)
	{
		retVal.objectiveMaps[i] = new NavObjectiveMap();
		retVal.objectiveMaps[i].objRef = nodeRef.objectiveMaps[i].objRef.id;
		retVal.objectiveMaps[i].read = nodeRef.objectiveMaps[i].read;
		retVal.objectiveMaps[i].write = nodeRef.objectiveMaps[i].write;
	}
	
	retVal.navStateRules = new Array();
	for (var i=0;i<nodeRef.navStateRules.length;i++)
	{
		retVal.navStateRules[i] = nodeRef.navStateRules[i].clone();
	}
	
	retVal.postConditionRules = new Array();
	for (var i=0;i<nodeRef.postConditionRules.length;i++)
	{
		retVal.postConditionRules[i] = nodeRef.postConditionRules[i].clone();
	}
	
	retVal.cFlow = nodeRef.cFlow;
	retVal.cChoice = nodeRef.cChoice;
	retVal.cForwardOnly = nodeRef.cForwardOnly;
	
	retVal.sRandomize = nodeRef.sRandomize;
	retVal.sSelect = nodeRef.sSelect;
	retVal.sSelectEvent = nodeRef.sSelectEvent;
	
	retVal.events = new EventsObject(DEFAULT_NODE_EVENTS);
	retVal.events.owner = retVal;
	retVal.navEngineRef = null;

	return retVal;
}

function NavEngineResolveClonedNodeReferencesRec(navRef, nodeRef)
{
	for (var i=0;i<nodeRef.parentStack.length;i++)
	{
		nodeRef.parentStack[i] = navRef.getNodeById(nodeRef.parentStack[i]);
	}

	for (var i=0;i<nodeRef.selectedChildren.length;i++)
	{
		nodeRef.selectedChildren[i] = navRef.getNodeById(nodeRef.selectedChildren[i]);
	}

	for (var i=0;i<nodeRef.children.length;i++)
	{
		this.resolveClonedNodeReferencesRec(navRef, nodeRef.children[i]);
	}
}

function NavEngineGetIllegalNavMessage()
{
	return this.illegalNavMessage;
}

function NavEngineSetIllegalNavMessage(msg)
{
	this.illegalNavMessage = msg;
}

function NavEngineGetNodeTag(id)
{
	var nodeRef = this.getNodeById(id);

	if (nodeRef != null)
	{
		return nodeRef.tag;
	}
}

function NavEngineGetNodeTitle(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.title;
	}
}

function NavEngineGetNodeChildIds(id)
{
	var nodeRef = this.getNodeById(id);
	var retVal = null;
	if (nodeRef != null)
	{
		retVal = new Array();
		var childArray = nodeRef.children;
		if (nodeRef.selectedChildren.length > 0)
		{
			childArray = nodeRef.selectedChildren;
		}
		for (var i=0;i<childArray.length;i++)
		{
			retVal[retVal.length] = childArray[i].id;
		}
	}
	
	return retVal;
}

function NavEngineGetNodeUrl(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.url;
	}
}

function NavEngineGetNodeId(nodeRef)
{
	if (nodeRef != null)
	{
		return nodeRef.id;
	}
	else
	{
		return null;
	}
}

function NavEngineGetNavWindow()
{
	return this.navWin;
}

function NavEngineSetNavWindow(winRef)
{
	if (this.navWin != winRef)
	{
		this.navWin = winRef;
		this.events.raiseEvent("navwindow",this);
	}
	return this.navWin;
}

function NavEngineSetNodeVisible(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setVisible(flag);
	}
}

function NavEngineGetNodeVisible(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getVisible();
	}
}

function NavEngineSetNodeEnabled(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setEnabled(flag);
	}
}

function NavEngineGetNodeEnabled(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getEnabled();
	}
}

function NavEngineSetNodeBlock(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setBlock(flag);
	}
}

function NavEngineGetNodeBlock(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getBlock();
	}
}

function NavEngineSetNodeSkip(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setSkip(flag);
	}
}

function NavEngineGetNodeSkip(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getSkip();
	}
}

function NavEngineSetNodeChoice(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setChoice(flag);
	}
}

function NavEngineGetNodeChoice(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getChoice();
	}
}

function NavEngineSetNodeFlow(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setFlow(flag);
	}
}

function NavEngineGetNodeFlow(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getFlow();
	}
}

function NavEngineSetNodeForwardOnly(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setForwardOnly(flag);
	}
}

function NavEngineGetNodeForwardOnly(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getForwardOnly();
	}
}

function NavEngineSetNodeSatisfied(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setSatisfied(flag);
	}
}

function NavEngineGetNodeSatisfied(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getSatisfied();
	}
}

function NavEngineSetNodeCompleted(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setCompleted(flag);
	}
}

function NavEngineGetNodeCompleted(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getCompleted();
	}
}

function NavEngineSetNodeAttempted(id, flag)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setAttempted(flag);
	}
}

function NavEngineGetNodeAttempted(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getAttempted();
	}
}

function NavEngineSetNodeScore(id, score)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.setScore(score);
	}
}

function NavEngineGetNodeScore(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		return nodeRef.getScore();
	}
}

function NavEngineResetNode(id)
{
	var nodeRef = this.getNodeById(id);
	if (nodeRef != null)
	{
		nodeRef.reset();
	}
}

function NavEngineGetChoice()
{
	return this.cChoice;
}

function NavEngineGetFlow()
{
	return this.cFlow;
}

function NavEngineGetForwardOnly()
{
	return this.cForwardOnly;
}

function NavEngineGetCurrentNode()
{
	return this.currentNode;
}

function NavEngineSetCurrentNode(nodeRef)
{
	if (nodeRef != this.getCurrentNode())
	{
		if (nodeRef != null && nodeRef.parentStack.length > 0)
		{
			if (nodeRef.parentStack[0].cChoice != this.cChoice)
			{
				this.cChoice = nodeRef.parentStack[0].cChoice;
				this.events.raiseEvent("choice",this);
			}

			if (nodeRef.parentStack[0].cFlow != this.cFlow)
			{
				this.cFlow = nodeRef.parentStack[0].cFlow;
				this.events.raiseEvent("flow",this);
			}

			if (nodeRef.parentStack[0].cForwardOnly != this.cForwardOnly)
			{
				this.cForwardOnly = nodeRef.parentStack[0].cForwardOnly;
				this.events.raiseEvent("forwardOnly",this);
			}
		}
	
		this.currentNode = nodeRef;
		this.events.raiseEvent("navchange",this);
		this.events.raiseEvent("navstatechange",this);
	}
}

function NavEngineCallbackNodeAttempted(ref)
{
	this.events.raiseEvent("nodeattempted",this,ref);
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeCompleted(ref)
{
	this.events.raiseEvent("nodecompleted",this,ref);
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeScore(ref)
{
	this.events.raiseEvent("nodescore",this,ref);
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeSatisfied(ref)
{
	this.events.raiseEvent("nodesatisfied",this,ref);
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeBlock(ref)
{
	this.events.raiseEvent("nodeblock",this,ref);
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeSkip(ref)
{
	this.events.raiseEvent("nodeskip",this,ref);
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeChoice(ref)
{
	var choiceChange = (this.currentNode != null && !this.isRootNode(this.currentNode) && this.currentNode.parentStack[0] == ref);
	if (choiceChange) this.cChoice = ref.cChoice;
	this.events.raiseEvent("nodechoice",this,ref);
	if (choiceChange)
	{
		this.events.raiseEvent("choice",this);
	}
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeFlow(ref)
{
	var flowChange = (this.currentNode != null && !this.isRootNode(this.currentNode) && this.currentNode.parentStack[0] == ref);
	if (flowChange) this.cFlow = ref.cFlow;
	this.events.raiseEvent("nodeflow",this,ref);
	if (flowChange)
	{
		this.events.raiseEvent("flow",this);
	}
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeForwardOnly(ref)
{
	var forwardOnlyChange = (this.currentNode != null && !this.isRootNode(this.currentNode) && this.currentNode.parentStack[0] == ref);
	if (forwardOnlyChange) this.cForwardOnly = ref.cForwardOnly;
	this.events.raiseEvent("nodeforwardonly",this,ref);
	if (forwardOnlyChange)
	{
		this.events.raiseEvent("forwardonly",this);
	}
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeEnabled(ref)
{
	this.events.raiseEvent("nodeenabled",this,ref);
	this.events.raiseEvent("navstatechange",this,ref);
}

function NavEngineCallbackNodeVisible(ref)
{
	this.events.raiseEvent("nodevisible",this,ref);
}

function NavEngineIsRootNode(nodeRef)
{
	for (var i=0;i<this.forest.length;i++)
	{
		if (nodeRef == this.forest[i])
		{
			return true;
		}
	}
	
	return false;
}

function NavEngineClearParentStacks()
{
	for (var i=0;i<this.forest.length;i++)
	{
		this.forest[i].parentStack.length = 0;
		this.clearParentStacksRec(this.forest[i]);
	}
}


function NavEngineClearParentStacksRec(nodeRef)
{
	if (nodeRef.parentStack.length > 1)
	{
		nodeRef.parentStack.length = 1;
	}
	
	for (var i=0;i<nodeRef.children.length;i++)
	{
		this.clearParentStacksRec(nodeRef.children[i]);
	}
}

function NavEngineCanNav(direction)
{
	return (this.evalNav(direction) == NAV_DENY_NONE);
}

function NavEngineCanGotoNode(id)
{
	return (this.evalGotoNode(id) == NAV_DENY_NONE);
}

function NavEngineFirstNodeInSequence(ptrRootNode)
{
	if (ptrRootNode == null) ptrRootNode = this.forest[0];
	return this.seqEngine.firstNodeInSequence(ptrRootNode);
}

function NavEngineNextNodeInSequence(ptrCurrentNode, bolAdjust)
{
	if (ptrCurrentNode == null) ptrCurrentNode = this.getCurrentNode();
	return this.seqEngine.nextNodeInSequence(ptrCurrentNode, bolAdjust);
}

function NavEnginePreviousNodeInSequence(ptrCurrentNode, bolAdjust)
{
	if (ptrCurrentNode == null) ptrCurrentNode = this.getCurrentNode();
	return this.seqEngine.previousNodeInSequence(ptrCurrentNode, bolAdjust);
}

// Takes a node Reference or a forest index.
function NavEngineGetTreeHash(forestIndex)
{
	var hashTable = new HashTable(NODE_HASH_LENGTH);
	
	if (typeof(forestIndex) == "number")
	{
		if (forestIndex > this.forest.length-1)
		{
			return null;
		}
		forestIndex = this.forest[index];
	}
	
	this.getTreeHashRec(hashTable, forestIndex);
	
	return hashTable;
}

function NavEngineGetTreeHashRec(hashRef, nodeRef)
{
	hashRef.insert(nodeRef.id, nodeRef);
	for (var i=0;i<nodeRef.children.length;i++)
	{
		this.getTreeHashRec(hashRef, nodeRef.children[i]);
	}
}

// Takes a node Reference or a forest index.
function NavEngineGetBlockedNodes(forestIndex)
{
	var arr = new Array();
	
	if (typeof(forestIndex) == "number")
	{
		if (forestIndex > this.forest.length-1)
		{
			return null;
		}
		forestIndex = this.forest[index];
	}
	
	this.getBlockedNodesRec(arr, forestIndex);
	
	return arr;
}

function NavEngineGetBlockedNodesRec(arr, nodeRef)
{
	if (nodeRef.block)
	{
		arr[arr.length] = nodeRef;
	}
	for (var i=0;i<nodeRef.children.length;i++)
	{
		this.getBlockedNodesRec(arr, nodeRef.children[i]);
	}
}

function NavEnginePopulateQuickReferenceContainers(nodeRef, enabled, rootNodeRef, treeIndex)
{
	if (!nodeRef)
	{
		this.htAncestorsEnabled = new HashTable();
		this.htNodeToRoot = new HashTable();
		this.arrBlockedNodes = new Array();
		for (var i=0;i<this.forest.length;i++)
		{
			this.arrBlockedNodes[i] = new Array();
			this.populateQuickReferenceContainers(this.forest[i], true, this.forest[i], i);
		}
		this.quickReferenceContainersDirty = false;
	}
	else
	{
		this.htAncestorsEnabled.insert(nodeRef.id, enabled);
		if (enabled && !nodeRef.enabled) enabled = false;
		this.htNodeToRoot.insert(nodeRef.id, rootNodeRef);
		if (nodeRef.blocked)
		{
			this.arrBlockedNodes[treeIndex][this.arrBlockedNodes[treeIndex].length] = nodeRef;
		}

		for (var i=0;i<nodeRef.children.length;i++)
		{
			this.populateQuickReferenceContainers(nodeRef.children[i], enabled, rootNodeRef, treeIndex);
		}
	}
}

function NavEngineNavStateChangeCallBack()
{
	this.quickReferenceContainersDirty = true;
}

function NavEngineEvalGotoNode(id)
{
	// Validate page id
	var nodeRef = this.getNodeById(id);	
	if (nodeRef == null)
	{
		return NAV_DENY_INVALID_ID;
	}
	
	// check to see if the node is enabled
	if (!nodeRef.getEnabled())
	{
		return NAV_DENY_DISABLED;
	}
	
/////////////////////
/////////////////////
/////////////////////
/////////////////////
	if (this.quickReferenceContainersDirty) this.populateQuickReferenceContainers();

	var ptrNext = nodeRef.parentStack[0];
	if (ptrNext != null)
	{
		// check to see if the node's parent
		// is enabled
		if (!ptrNext.getEnabled())
		{
			return NAV_DENY_PARENT_DISABLED;
		}

		// check to see if all ancestors are enabled
		if (!this.htAncestorsEnabled.retrieve(ptrNext.id)) return NAV_DENY_ANCESTOR_DISABLED;
		
/*		ptrNext = ptrNext.parentStack[0];
		while (ptrNext != null)
		{
			if (!ptrNext.getEnabled())
			{
				return NAV_DENY_ANCESTOR_DISABLED;
			}
			ptrNext = ptrNext.parentStack[0];
		}
*/
	}

/*	// Locate root of the tree that the requested
	// node resides in.
	ptrNext = nodeRef.parentStack[0];
	var root = nodeRef;
	while (ptrNext != null)
	{
		root = ptrNext;
		ptrNext = ptrNext.parentStack[0];
	}
*/
////////////////////////////////////////////////////////////////////////////
	// Get array filled with all blocked nodes in the
	// same tree as the requested node.
/*	var keys = this.getBlockedNodes(root);*/
//////////////////////////////////////////////////////////////////////////////

	// Search through all blocked nodes in the same tree, looking for any
	// that are blocked. If a blocked node is found
	// then the requested node's num is checked against
	// the num of the blocked node's next sibling.
	var rootNodeRef = this.htNodeToRoot.retrieve(nodeRef.id);
	var forestIndex = null;
	for (var i=0;i<this.forest.length;i++)
	{
		if (this.forest[i].id == rootNodeRef.id)
		{
			forestIndex = i;
			break;
		}
	}
	
	for (var i=0;i<this.arrBlockedNodes[forestIndex].length;i++)
	{
		// if the blocked node has a sibling following it
		if (this.arrBlockedNodes[forestIndex][i].getNextSibling(true) != null && this.arrBlockedNodes[forestIndex][i].parentStack[0] != null)
		{
			var lastNodeInSubtree = this.getLastNodeInSubtree(this.arrBlockedNodes[forestIndex][i].parentStack[0]);
			// If the number of the selected node is >= the blocked node's next sibling's
			// number and <= the number of the last node in the blocked node's parent's subtree
			if (nodeRef.num >= this.arrBlockedNodes[forestIndex][i].getNextSibling(true).num && nodeRef.num <= lastNodeInSubtree.num)
			{
				this.blockedNode = this.arrBlockedNodes[forestIndex][i];
				return NAV_DENY_BLOCKED;
			}
		}
	} 
/////////////////////
/////////////////////
/////////////////////
/////////////////////
	if (this.firstNodeInSequence(nodeRef) == null)
	{
		return NAV_DENY_EXHAUSTED;
	}
	
	if (this.getCurrentNode() != null)
	{
		// Check to see if the parent node has cForwardOnly turned on
		if (this.getCurrentNode().parentStack.length > 0 && this.getCurrentNode().parentStack[0].cForwardOnly)
		{
			var parentNode = this.getCurrentNode().parentStack[0];
			// First check to see if the destination is contained inside the
			// currentNode's parent.
			var nextSib = parentNode.getNextSibling(true);
			if ((nextSib == null && nodeRef.num > parentNode.num) || (nextSib != null && (nodeRef.num > parentNode.num && nodeRef.num < parentNode.getNextSibling(true).num)))
			{
				// The destination node is inside the currentNode's parent,
				// so now check to see if the destination node has a smaller
				// num property.
				if (nodeRef.num < this.getCurrentNode().num)
				{
					this.blockedNode = this.getCurrentNode();
					return NAV_DENY_FORWARD_ONLY;
				}
			}
		}
	}
	
	return NAV_DENY_NONE;
}

function NavEngineGetLastNodeInSubtree(root)
{
	var children = root.children;
	if (root.selectedChildren.length > 0)
	{
		children = root.selectedChildren;
	}
	
	if (children.length == 0)
	{
		return root;
	}
	else
	{
		return this.getLastNodeInSubtree(root.children[children.length-1]);
	}
}

function NavEngineGosubNode(id, retId)
{
	if (retId == null)
	{
		if (this.getCurrentNode() != null)
		{
			retId = this.getCurrentNode().id;
		}
	}
	
	if (retId != null)
	{
		var retNodeRef = this.getNodeById(retId);
		if (this.evalGotoNode(id) == NAV_DENY_NONE && retNodeRef != null)
		{
			var nodeRef = this.getNodeById(id);
			nodeRef.parentStack[nodeRef.parentStack.length] = retNodeRef;
			
			this.setCurrentNode(this.firstNodeInSequence(nodeRef));
		}
		else
		{
			this.setCurrentNode(null);
		}
		this.showCurrent();
	}
	else
	{
		this.gotoNode(id);
	}
}

function NavEngineGotoNode(id)
{
	if (this.evalGotoNode(id) == NAV_DENY_NONE)
	{
		var nodeRef = this.getNodeById(id);
		
		// The loop below searches upwards through the tree, begining with
		// the current node, looking for the first parent node that has at
		// least one extra parent in its parentStack array (signifying that
		// the node has an outstanding gosub operation). 
		var ptr = this.getCurrentNode();
		while (ptr != null)
		{
			// Find the first ancestor node with a parentStack that holds a gosub return value.
			if ( (this.isRootNode(ptr) && ptr.parentStack.length > 0) || (!this.isRootNode(ptr) && ptr.parentStack.length > 1) )
			{
				// A node was found, so we will test the destination node's num
				// against ptr to see if it lies outside ptr's subtree. If it lies
				// outside the subtree, then any outstanding gosub call is rendered
				// invalid, and all parentStacks in the forest will need to be cleared.
				// (We use the nodes' num properties to detect their relative positions)
				if (ptr.num > nodeRef.num)
				{
					this.clearParentStacks();
				}
				else if (ptr.getNextSibling() == null)
				{
					if (nodeRef.num > this.getLastNodeInSubtree(ptr).num)
					{
						this.clearParentStacks();
					}
				}
				else
				{
					if (nodeRef.num >= ptr.getNextSibling().num)
					{
						this.clearParentStacks();
					}
				}
				// Since the first ancestor found will always be the lowest
				// in the tree, we do not need to search up the tree any further.
				break;
			}
			ptr = ptr.parentStack[0];
		}
		this.setCurrentNode(this.firstNodeInSequence(nodeRef));
	}
	else
	{
		this.setCurrentNode(null);
	}
	
	this.showCurrent();
}

var _NavUrlStack = new Array();
var _NavWinStack = new Array();
function NavEngineShowCurrent()
{
	if (this.navWin != null && !this.navWin.closed)
	{
		if (this.getCurrentNode() == null)
		{
			// Handle what to display if no current node
			// Handle displaying current node
			_Splice(_NavUrlStack,0,0,"naverror.html");
			_Splice(_NavWinStack,0,0,this.navWin);
			setTimeout("_Pop(_NavWinStack).location = _Pop(_NavUrlStack)",NAV_DELAY);
		}
		else
		{
			// Handle displaying current node
			_Splice(_NavUrlStack,0,0,this.getCurrentNode().url);
			_Splice(_NavWinStack,0,0,this.navWin);
			setTimeout("_Pop(_NavWinStack).location = _Pop(_NavUrlStack)",NAV_DELAY);
		}
	}
}

function NavEngineShowNodeInWindow(id, winRef)
{
	if (winRef != null && !winRef.closed)
	{
		if (this.getNodeById(id) == null)
		{
			// Handle what to display if no current node
			winRef.document.open();
			winRef.document.writeln("Navigation Fault: specified page is null");
			winRef.document.close();
		}
		else
		{
			// Handle displaying current node
			_Splice(_NavUrlStack,0,0,this.getNodeById(id).url);
			_Splice(_NavWinStack,0,0,winRef);
			setTimeout("_Pop(_NavWinStack).location = _Pop(_NavUrlStack)",NAV_DELAY);
		}
	}
}

function NavEngineInitNav()
{
	var nodeRef = null;
	for (var i=0;i<this.forest.length;i++)
	{
		if (this.forest[i].getEnabled())
		{
			nodeRef = this.firstNodeInSequence(this.forest[i]);
			if (nodeRef != null)
			{
				break;
			}
		}	
	}
	this.setCurrentNode(nodeRef);
}

function NavEngineDoNav(direction)
{
	var objVal = null;
	var prevNode = this.getCurrentNode();
	switch (direction)
	{
		case NAV_DIRECTION_FORWARD:objVal=this.seqEngine.nextInSequence(this.getCurrentNode(),true);break;
		case NAV_DIRECTION_REVERSE:objVal=this.seqEngine.previousInSequence(this.getCurrentNode(),true);break;
	}
	
	if (objVal == null)
	{
		this.setCurrentNode(null);
	}
	else
	{
		// Check to see if this is a gosub operation.  If it is,
		// then we have to validate the operation and resolve the final
		// destination node.
		if (objVal.returnLocation != null)
		{
			var destNode = null;
			// Validate the operation
			if (this.evalGotoNode(objVal.destination.id) == NAV_DENY_NONE)
			{
				// Operation is valid, so resolve final destination
				destNode = this.firstNodeInSequence(objVal.destination);
				if (destNode != null)
				{
					// Final destination is valid, so push a reference 
					// to the currentNode on the target destination's
					// parentStack array.
					objVal.destination.parentStack[objVal.destination.parentStack.length] = objVal.returnLocation;
				}
			}
			this.setCurrentNode(destNode);
		}
		else
		{
			// Since this is not a gosub operation,
			// we can assume that objVal.destination is
			// a valid destination (seqEngine.nextInSequence()
			// ans seqEngine.previousInSequence() perform all
			// navigation resolution & validation on standard
			// next/back navigation operations).
			this.setCurrentNode(objVal.destination);
		}
	}
	
	this.showCurrent();
}

function NavEngineGetNav(direction)
{
	var objRef = null;
	var retVal = null;
	switch (direction)
	{
		case NAV_DIRECTION_FORWARD:objRef=this.seqEngine.nextInSequence(this.getCurrentNode());break;
		case NAV_DIRECTION_REVERSE:objRef=this.seqEngine.previousInSequence(this.getCurrentNode());break;
	}

	if (objRef != null)
	{
		if (objRef.returnLocation != null)
		{
			if (this.evalGotoNode(objRef.destination.id) == NAV_DENY_NONE)
			{
				// Operation is valid, so resolve final destination
				retVal = this.firstNodeInSequence(objRef.destination);
			}
		}
		else
		{
			retVal = objRef.destination;
		}
	}
	
	return retVal;
}

function NavEngineEvalNav(direction)
{
	var retVal = this.getNav(direction);
	if (retVal == null)
	{
		if (this.blockedNode == null)
		{
			if (direction == NAV_DIRECTION_FORWARD)
			{
				return NAV_DENY_BLOCKED;
			}
			else
			{
				return NAV_DENY_FORWARD_ONLY;
			}
		}
		else
		{
			return NAV_DENY_EXHAUSTED;
		}
	}
	else
	{
		return NAV_DENY_NONE;
	}
}

function NavEngineInitForest()
{
	for (var i=0;i<this.forest.length;i++)
	{
		this.forest[i].init();
	}
}

function NavEngineResetForest()
{
	for (var i=0;i<this.forest.length;i++)
	{
		this.forest[i].reset();
	}
}

function NavEngineGetObjectiveById(id)
{
	return this.objHash.retrieve(id.toLowerCase());
}

function NavEngineGetNodeById(id)
{
	var retVal = this.nodes.retrieve(id);
	if (retVal == null)
	{
		retVal = this.taggedNodes.retrieve(id);
	}
	return retVal;
}

function NavEngineGetNodeByNum(num)
{
	var retVal = this.numberedNodes.retrieve(num);
	return retVal;
}

function NavEngineToString(num)
{
	var strTab = "";
	var str = "";
	
	if (num == null) num = 0;
	for (var i=0;i<num;i++) strTab += "\t";
	
	str = "NavEngine:";
	
	var strTemp = this.dump();
	str += "\n" + strTab + "\tDump (length='" + strTemp.length + "'): \n" + strTab + "\t" + strTemp;

	str += "\n" + strTab + "\tcurrentNode: " + ((this.getCurrentNode() != null)?(this.getCurrentNode().id):("null"));
	
	str += "\n" + strTab + "\tobjectives: ";
	for (var i=0;i<this.objectives.length;i++)
	{
		str += this.objectives[i].toString(num) + ((i<this.objectives.length-1)?(","):(""));
	}

	str += "\n" + strTab + "\tobjHash: " + this.objHash.toString();

	str += "\n" + strTab + "\tnodes: " + this.nodes.toString();
	
	str += "\n" + strTab + "\tforest: ";
	for (var i=0;i<this.forest.length;i++)
	{
		str += this.forest[i].toString(num) + ((i<this.forest.length-1)?(","):(""));
	}
	
	str += "\n" + strTab + "\ttaggedNodes: " + this.taggedNodes.toString();

	str += "\n" + strTab + "\tnumberedNodes: " + this.numberedNodes.toString();

	return str;
}

function NavEngineSetDefaultValues()
{
	this.cChoice = true;
	this.cFlow = true;
	this.cForwardOnly = false;
	this.seqEngine = new NavSeqEngine(this);
	this.mapAPI = new NavMapAPI(this);
	this.nodeAPI = new NavNodeAPI(this);
	this.forest = new Array();
	this.nodes = new HashTable(NODE_HASH_LENGTH);
	this.taggedNodes = new HashTable(NODE_HASH_LENGTH);
	this.numberedNodes = new HashTable(NODE_HASH_LENGTH);
	this.objectives = new Array();
	this.objHash = new HashTable(OBJ_HASH_LENGTH);
	this.currentNode = null;
	this.blockedNode = null;
	this.navWin = null;
	this.illegalNavMessage = DEFAULT_ILLEGAL_NAV_MESSAGE;
	this.loadStringSeparator = DEFAULT_LOAD_STRING_SEPARATOR;
	this.title = "";
	this.description = "";
}

function NavEngineLoadFromXMLDoc(xmlDoc)
{	
	this.setDefaultValues();

	// Load global objectives
	var objectivesElement = xmlDoc.docNode.getElements("objectives");
	if (objectivesElement.length >0)
	{
		var objectivesNode = objectivesElement[0];
		var objNodes = objectivesNode.getElements("objective");
		for (var i=0;i<objNodes.length;i++)
		{
			var objRef = new NavObjective();
			objRef.loadFromXMLNode(objNodes[i]);
			this.objHash.insert(objRef.id, objRef);
			this.objectives[this.objectives.length] = objRef;
		}
	}
	var titleNode = (xmlDoc.docNode.getElements("title"))[0];
	if (titleNode != null)
	{
		this.title = titleNode.getText();
	}
	else
	{
		this.title = "";
	}
	
	var descriptionNode = (xmlDoc.docNode.getElements("description"))[0];
	if (descriptionNode != null)
	{
		this.description = descriptionNode.getText();
	}
	else
	{
		this.description = "";
	}

	// Load lesson nodes
	var forestNode = (xmlDoc.docNode.getElements("forest"))[0];
	var objNodes = forestNode.getElements("node");
	for (var i=0;i<objNodes.length;i++)
	{
		var nodeRef = new NavNode();
		nodeRef.loadFromXMLNode(objNodes[i], this);
		this.forest[this.forest.length] = nodeRef;
	}

	this.populateNodesTable();
	this.numberNodes();
	
	// Resolve forest's internal references
	for (var i=0;i<this.forest.length;i++)
	{
		this.forest[i].resolveReferences(this);
	}
	
	// The tree must be initialized before
	// it can be used.
	this.initForest();
	this.resetForest();
}

function NavEngineRegisterNodeEvents()
{
	for (var i=0;i<this.forest.length;i++)
	{
		this.registerNodeEventsRec(this.forest[i]);
	}
}

function NavEngineRegisterNodeEventsRec(nodeRef)
{
	nodeRef.events.registerEvent("reorder","callbackReorder",this,null,true);
	nodeRef.events.registerEvent("attempted","callbackNodeAttempted",this,null,true);
	nodeRef.events.registerEvent("completed","callbackNodeCompleted",this,null,true);
	nodeRef.events.registerEvent("satisfied","callbackNodeSatisfied",this,null,true);
	nodeRef.events.registerEvent("score","callbackNodeScore",this,null,true);
	nodeRef.events.registerEvent("block","callbackNodeBlock",this,null,true);
	nodeRef.events.registerEvent("skip","callbackNodeSkip",this,null,true);
	nodeRef.events.registerEvent("enabled","callbackNodeEnabled",this,null,true);
	nodeRef.events.registerEvent("visible","callbackNodeVisible",this,null,true);
	nodeRef.events.registerEvent("choice","callbackNodeChoice",this,null,true);
	nodeRef.events.registerEvent("flow","callbackNodeFlow",this,null,true);
	nodeRef.events.registerEvent("forwardOnly","callbackNodeForwardOnly",this,null,true);
	for (var i=0;i<nodeRef.children.length;i++)
	{
		this.registerNodeEventsRec(nodeRef.children[i]);
	}
}

// This is a callback that gets triggered
// when a node reorders it's children
function NavEngineCallbackReorder(ref)
{
	this.numberNodes();
	this.events.raiseEvent("reorder",this,ref);
	this.events.raiseEvent("navstatechange",this,ref);	
}

var NODE_COUNT = 0;
// Used to assign num values to nodes, using
// the order that the nodes are arranged in the
// nodes' selectedChildren[] array.
function NavEngineNumberNodes()
{
	NODE_COUNT = 0;
	this.numberedNodes = new HashTable(NODE_HASH_LENGTH);
	for (var i=0;i<this.forest.length;i++)
	{
		this.numberNodesRec(this.forest[i], [], i+1);
	}
}

function NavEngineNumberNodesRec(nodeRef, positionArray, positionNum)
{
	var nextNum = 1;
	var posArray = positionArray.concat();
	nodeRef.num = NODE_COUNT++;
	this.numberedNodes.insert(nodeRef.num, nodeRef);
	
	// set num property of all children to null
	for (var i=0;i<nodeRef.children.length;i++)
	{
		nodeRef.children[i].num = null;
	}

	// Determine which node child array to use (children or selectedChildren)
	var childArray = nodeRef.children;
	if (nodeRef.selectedChildren.length > 0)
	{
		childArray = nodeRef.selectedChildren;
	}
	
	// Recurse on the proper node child array.
	if (nodeRef.transparent && nodeRef.parentStack[0] != null)
	{
		for (var i=0;i<childArray.length;i++)
		{
			positionNum = this.numberNodesRec(childArray[i], posArray, positionNum);
		}
	}
	else
	{
		posArray[posArray.length] = positionNum++;
		for (var i=0;i<childArray.length;i++)
		{
			nextNum = this.numberNodesRec(childArray[i], posArray, nextNum);
		}
	}
	
	// if any children still have null num property
	// then they (and their children) will be assigned
	// numbers that come after their selected siblings.
	// This step ensures that all nodes in the forest
	// have a valid num property - event those that
	// aren't currently selected as children.
	if (nodeRef.transparent && nodeRef.parentStack[0] != null)
	{
		var num = positionNum;
		for (var i=0;i<nodeRef.children.length;i++)
		{
			if (nodeRef.children[i].num == null)
			{
				num = this.numberNodesRec(nodeRef.children[i], posArray, num);
			}
		}
	}
	else
	{
		var num = nextNum;
		for (var i=0;i<nodeRef.children.length;i++)
		{
			if (nodeRef.children[i].num == null)
			{
				num = this.numberNodesRec(nodeRef.children[i], posArray, num);
			}
		}
	}
	
	nodeRef.position = posArray;
	return positionNum;
}

// Used to populate the nodes and tagged nodes
// hash tables with all nodes in the forest.
function NavEnginePopulateNodesTable()
{
	for (var i=0;i<this.forest.length;i++)
	{
		this.populateNodesTableRec(this.forest[i]);
	}
}

function NavEnginePopulateNodesTableRec(nodeRef)
{
	this.nodes.insert(nodeRef.id,nodeRef);
	if (nodeRef.tag != "")
	{
		this.taggedNodes.insert(nodeRef.tag,nodeRef);
	}
	for (var i=0;i<nodeRef.children.length;i++)
	{
		this.populateNodesTableRec(nodeRef.children[i]);
	}
}

